define([
    /* Core */
    'App',
    'Vent',
    'User',
    'modules/appointments/appointments-module',
    'underscore',
    'backbone',
    'marionette',

    /* Utilities */
    'modules/appointments/appointments-radio',
    'modules/appointments/appointments-module',
    'modules/notification-preferences/notification-preferences-module',
    'modules/form/form-view-mixin',

    /* Views */
    'modules/new-appointment-request/views/common/header/view',
    'modules/new-appointment-request/views/common/error/view',
    'modules/new-appointment-request/views/common/email/section-view',
    'modules/new-appointment-request/views/common/cancel-modal/popup-view',
    'modules/new-appointment-request/views/common/confirmation-modal/popup-view',
    'modules/form/leave-confirmation/leave-confirmation-popup-view',

    /* Resources */
    'modules/new-appointment-request/resources/common/model',
    'modules/new-appointment-request/resources/direct/model',
    'modules/new-appointment-request/resources/request/model',
    'modules/appointments/messages/request-message-view',

    /* Templates */
    'text!modules/new-appointment-request/template.html',
], function(/* Core */
    app,
    vent,
    User,
    module,
    _,
    Backbone,
    Marionette,
    /* Utilities */
    radio,
    appointmentModule,
    notificationModule,
    Mixin,
    /* Views */
    HeaderView,
    ErrorView,
    EmailPreferencesView,
    CancelConfirmationModal,
    ConfirmAppointmentPopupView,
    LeaveConfirmationPopupView,
    /* Resources */
    AppointmentModel,
    DirectModel,
    RequestModel,
    MessageView,
    /* Templates */
    template
) {
    'use strict';


    var SUBMIT_REQUEST = 'Submit Request';
    var SUBMIT_DIRECT = 'Schedule Appointment';
    var CONFIRM_DIRECT = 'Confirm Appointment';
    var RESPONSE_SUCCESS = 200;


    return Marionette.View.extend(Mixin.form({
        template: _.template(template),

        className: 'appointments',

        ui: {
            cancel: '#new-appointment-request-cancel',
            submit: '#schedule-appointment-btn',
            confirm: '#confirm-appointment-btn',
        },

        events: {
            'click @ui.cancel': 'onCancel',
            'click @ui.submit': 'onSubmit',
            'click @ui.confirm': 'onConfirm',
        },

        regions: {
            header: '.new-appointments-header',
            steps: '.new-appointments-steps',
            error: '.new-appointments-errors',
            email: '.new-appointments-email',
        },

        childViewEvents: {
            'show:error': 'onServerError',
            'navigate:appointments': 'routeHome',
            'hide:email': 'resetEmailRegion',
        },

        modelEvents: {
            'change:clinic': 'showEmailRegion',
            'change:scheduling-method': 'hideButton onMethodChange',
            'change': 'resetErrorRegion',
            'change:typeOfCare': 'hideButton',
            'change:facility': 'hideButton',
        },

        initialize: function(options) {
            this.model = new AppointmentModel();
            this.listenTo(vent, 'wizard:exit', this.handleExitWizard);
            this.isTelehealthFeaturesEnabled = _.isUndefined(app.disabledFeatures.get('TH_FEATURE_SET'));
            this.googleAnalytics = options.googleAnalyticsProvider.getInstance();
            this.schedulingStepsLayoutProvider = options.schedulingStepsLayoutProvider;
        },

        onRender: function() {
            var header = this.getRegion('header');

            this.initializeMixin();
            header.show(new HeaderView());
            this.showStepsRegion();
        },

        /**
         * Sets up the form with the Mixin Components
         * @return {void}
         */
        initializeMixin: function() {
            this.initializeFormValidation({
                ignore: ':hidden:not(#dateTime-hidden)',
                errorLabelContainer: '.error-list',
            });
            this.initializeFDNS   firmNavigation({
                $form: this.$el.find('form > :not(.facility-care-and-scheduling-section)'),

                // eslint-disable-next-line max-len
                warningMessage: 'You have not completed your appointment/request. If you wish to continue without completing, select CONTINUE, otherwise select RETURN to return to the New Appointments/Requests screen.',
            });
        },

        /**
         * @return {void}
         */
        showStepsRegion: function() {
            var steps = this.getRegion('steps');

            steps.show(this.schedulingStepsLayoutProvider.getInstance(this.model, this.validator));
        },

        /**
         * @param {Backbone.Collection} errorCollection
         * @return {void}
         */
        showErrorsRegion: function(errorCollection) {
            var error = this.getRegion('error');

            this.resetErrorRegion();
            error.show(new ErrorView({collection: errorCollection}));
        },

        /**
         * @return {void}
         */
        showEmailRegion: function() {
            var region = this.getRegion('email');
            this.emailModel = notificationModule.notificationPreferences.clone();

            region.show(new EmailPreferencesView({
                model: this.emailModel,
                isForExpressCare: this.model.isExpressCare(),
                isForVideoVisit: Boolean(this.isTelehealthFeaturesEnabled && this.model.isVideoVisit()),
            }));
        },

        /**
         * @return {void}
         */
        resetEmailRegion: function() {
            var email = this.getRegion('email');

            if (email.currentView) {
                email.reset();
            }
        },

        /**
         * @return {void}
         */
        resetErrorRegion: function() {
            var error = this.getRegion('error');

            if (error.currentView) {
                error.reset();
            }
        },

        /**
         * Called when the scheduling method is changed.
         * Used to set the correct name on the submit button
         * @return {void}
         */
        onMethodChange: function() {
            var buttonName = this.getSubmitButtonName();
            var schedulingMethod = this.model.isRequest() ? 'Submit a Request to a VA Scheduler' : 'Schedule Myself';

            this.googleAnalytics
                .gas('send', 'event', 'veteran-appointment', 'user-selected-scheduling-method', schedulingMethod);

            if (this.model.isRequest()) {
                this.showEmailRegion();
            }

            if (this.isTelehealthFeaturesEnabled && this.model.isDirect()) {
                this.ui.confirm.text(buttonName);
                this.ui.confirm.removeClass('hidden');
            } else {
                this.ui.submit.text(buttonName);
                this.ui.submit.removeClass('hidden');
                this.googleAnalytics.gas('send', 'event', 'veteran-appointment', 'appointment-request-requested');
            }

            
        },

        /**
         * Called when the submit button in request scheduling is pressed
         * @return {void}
         */
        onSubmit: function() {
            var isFormValid = this.$el.find('form').valid();

            this.resetErrorRegion();
            if (isFormValid) {
                // I think this would be better in the success function
                // but it is being put here because this is the equivalent spot to where digital services put it.
                if (typeof gas !== 'undefined') {
                    gas('send', 'event', 'veteran-appointment', 'appointment-scheduled-successful');
                }

                this.saveForm();
            } else {
                this.onFormErrors();
            }
        },

        /**
         * Called when the confirm button in direct scheduling is pressed
         * @return {void}
         */
        onConfirm: function() {
            var isFormValid = this.$el.find('form').valid();
            this.resetErrorRegion();

            if (isFormValid) {
                this.showConfirmationPopupView();
            } else {
                this.onFormErrors();
            }
        },

        showConfirmationPopupView: function() {
            var confirmAppointmentPopupView = new ConfirmAppointmentPopupView({
                model: this.model,
                typeOfCareName: this.model.getCareName(),
                notificationEmail: this.emailModel.get('emailAddress'),
                elAfterClose: this.$el.find('#confirm-appointment-btn'),
            });

            this.listenTo(confirmAppointmentPopupView, 'continue', function() {
                // I think this would be better in the success function
                // but it is being put here because this is the equivalent spot to where digital services put it.
                if (typeof gas !== 'undefined') {
                    gas('send', 'event', 'veteran-appointment', 'user-request-appointment-scheduled');
                }

                this.saveForm();
            }, this);

            this.listenToOnce(confirmAppointmentPopupView, 'destroy', function() {
                this.stopListening(confirmAppointmentPopupView, 'continue');
            }, this);

            confirmAppointmentPopupView.openPopup();
        },
        /**
         * @return {void}
         */
        saveForm: function() {
            var Model = this.model.isDirect() ? DirectModel : RequestModel;

            this.saveEmailPreferences();
            this.saveModel = new Model(this.model.attributes);
            this.listenToOnce(this.saveModel, 'save:success', this.onSaveSuccess);
            this.listenToOnce(this.saveModel, 'save:error', this.onSaveError);
            this.saveModel.save();
        },

        /**
         * @return {void}
         */
        saveEmailPreferences: function() {
            var notificationFrequency = notificationModule.notificationPreferences.get('emailAllowed') ?
                'Each new message' :
                'Never';
            var attributes;
            var user;

            notificationModule.notificationPreferences.set('notificationFrequency', notificationFrequency);
            attributes = this.emailModel.attributes;
            user = this.model.user();
            this.model.set('emailPreferences', this.emailModel);

            notificationModule.notificationPreferences.savePreferences(attributes, user);
        },

        /**
         * Called when the cancel button is pressed
         * @return {void}
         */
        onCancel: function() {
            var entryType;
            var model;
            var Popup;

            if (this.model.isRequest() || this.model.isDirect()) {
                entryType = this.model.isDirect() ? 'appointment' : 'request';
                model = new Backbone.Model({entryType: entryType});
                Popup = new CancelConfirmationModal({
                    elAfterClose: '#new-appointment-request-cancel-btn',
                    model: model,
                    onContinue: _.bind(this.navigateHome, this),
                });

                Popup.openPopup();
                return;
            }
            this.navigateHome();
        },

        /**
         * Displays Error Messages from within the form
         * @return {void}
         */
        onFormErrors: function() {
            var errors = _.map(this.validator.errorList, function(error) {
                return {error: error.message};
            });

            var collection = new Backbone.Collection(errors);

            this.showErrorsRegion(collection);

            if (typeof gas !== 'undefined') {
                if (this.model.isRequest()) {
                    gas('send', 'event', 'veteran-appointment', 'appointment-request-unsuccessful');
                } else if (this.model.isDirect()) {
                    gas('send', 'event', 'veteran-appointment', 'appointment-scheduled-fail');
                }
            }
        },

        /**
         * Displays errors returned from the server
         * @param {object} data Server Response Data
         * @return {void}
         */
        onServerError: function(data) {
            var json = data.responseJSON || {};
            var errors = json.errors || [
                {
                    error: 'There was an error saving your request to the server. ' +
                    'Please try again later.',
                },
            ];
            var collection = new Backbone.Collection(errors);

            this.showErrorsRegion(collection);
        },

        /**
         * @param {Backbone.Model} [model] The model returned on the request save event
         * @param {*} response
         * @return {void}
         */
        onSaveSuccess: function(model, response) {
            this.disableShowConfirmNavigation();
            if (this.model.isDirect()) {
                this.onAppointmentSaveSuccess();
                return;
            }
            this.onRequestSaveSuccess(response);
        },

        /**
         * Catches a fake error when the server returns no data
         * @param {Backbone.Model} model
         * @param {jQuery.xhr} response
         * @param {object} options
         * @return {void}
         */
        onSaveError: function(model, response, options) {
            if (response.status === RESPONSE_SUCCESS) {
                // Not a real error the server did not send anything
                // So it failed to parse the model
                this.onSaveSuccess(model, response, options);
                return;
            }
            this.onServerError(response);
        },

        /**
         * Updates the appointment module with the new item and then navigates to the
         * details page.
         * @return {void}
         */
        onAppointmentSaveSuccess: function() {
            var detailsModel = this.saveModel.toAppointmentsModel();

            appointmentModule.appointments.add(detailsModel);
            this.navigateToDirectDetails(detailsModel);
        },

        /**
         * Updates the request module with the new item and then navigates to the
         * details page.
         * @param {Backbone.Model} response The reply from the server
         * @return {void}
         */
        onRequestSaveSuccess: function(response) {
            var Model = appointmentModule.requests.model;
            var request = new Model(response, {parse: true});
            var messageView;
            // add AppointmentModel's .getUserId() to the requestModel through userId property
            request.set('userId', this.model.getUserId());
            messageView = new MessageView(request);
            appointmentModule.requests.add(request);

            messageView.checkForMessage(this.model);
        },

        /**
         * @param {Backbone.Model} detailsModel
         * @return {void}
         */
        navigateToDirectDetails: function(detailsModel) {
            var options = {showStatus: true, showFeedbackLink: true, showCancellation: false};

            // TODO This will move when i fix the getClinicName function
            var clinic = detailsModel.get('clinic');
            clinic.friendlyName = this.model.getFacilityName();
            clinic.name = this.model.getClinicName();

            radio.trigger('appointments:appointment:selected', detailsModel, options);
        },

        /**
         * @return {void}
         */
        navigateHome: function() {
            var entryType;

            if (typeof gas !== 'undefined') {
                entryType = (this.model.get('scheduling-method') === 'direct') ? 'appointment' : 'request';
                gas('send', 'event', 'veteran-appointment', 'user-cancelled-appointment-request', entryType);
            }
            this.disableShowConfirmNavigation();
            this.routeHome();
        },

        /**
         * @return {void}
         */
        routeHome: function() {
            window.location.hash = '#appointments';
        },

        /**
         * Hides submit and confirm buttons
         * @return {void}
         */
        hideButton: function() {
            this.ui.confirm.addClass('hidden');
            this.ui.submit.addClass('hidden');
        },

        /**
         * @return {string} The name to display on the submit button
         */
        getSubmitButtonName: function() {
            if (this.isTelehealthFeaturesEnabled) {
                return this.model.isRequest() ? SUBMIT_REQUEST : CONFIRM_DIRECT;
            }

            return this.model.isRequest() ? SUBMIT_REQUEST : SUBMIT_DIRECT;
        },

        handleExitWizard: function(e, callback) {
            // when: field is dirty and need to show exit modal
            // then: show exit modal - on the exit modal this needs to be able to invoke callback()
            //      when the user hits continue
            var leaveConfirmationPopup;

            if (this.model.get('scheduling-method')) {
                leaveConfirmationPopup = new LeaveConfirmationPopupView({
                    elAfterClose: $(e.currentTarget),
                    continue: callback,
                });
                leaveConfirmationPopup.openPopup();
            } else {
                callback();
            }
        },
    }));
});
